{- |
The modules in this module are equivalent to those in "Data.List.Key",
but they expect that the key function is only a selector
or otherwise a function that is cheap to compute.
It will be evaluated multiple times per element.
-}
module Data.List.Key.Selector where

import qualified Data.List.Comparing as Cmp
import Data.Function.HT (compose2)


{- $setup
>>> import qualified Data.List.Key as Key
>>>
>>> import qualified Test.QuickCheck.Modifiers as Mod
>>> import Test.QuickCheck ((===))
-}


aux ::
   ((a -> a -> b) -> [a] -> c) ->
      (key -> key -> b) -> (a -> key) ->
          ([a] -> c)
aux listFunc cmpFunc key =
   listFunc (compose2 cmpFunc key)


{- |
prop> \xs -> let key = fst in Key.group key xs === group key (xs :: [(Ordering,Bool)])
prop> \xs -> let key = snd in Key.group key xs === group key (xs :: [(Ordering,Bool)])

Alternatively you may write @groupBy ((==) `on` key)@.
-}
group :: Eq b => (a -> b) -> [a] -> [[a]]
group  =  aux Cmp.group (==)


{- |
prop> \(Mod.Ordered xs) (Mod.Ordered ys) -> merge fst xs ys === Key.merge fst xs (ys::[(Int,Char)])
-}
merge :: Ord b => (a -> b) -> [a] -> [a] -> [a]
merge key =
   let go kx x xs ky y ys =
         if kx <= ky
         then x : case xs of
            [] -> y : ys
            x1 : xs1 -> go (key x1) x1 xs1 ky y ys
         else y : case ys of
            [] -> x : xs
            y1 : ys1 -> go kx x xs (key y1) y1 ys1

       -- turn into top-level matches?
       start [] ys = ys
       start xs [] = xs
       start (x : xs) (y : ys) = go (key x) x xs (key y) y ys
   in start

{- |
prop> \(Mod.Ordered xs) (Mod.Ordered ys) -> mergeAlt fst xs ys === Key.merge fst xs (ys::[(Int,Char)])
-}
mergeAlt :: Ord b => (a -> b) -> [a] -> [a] -> [a]
mergeAlt key =
   let go _kx [] _ky ys = ys
       go _kx xs _ky [] = xs
       go kx xss@(x : xs) ky yss@(y : ys) =
         if kx <= ky
         then x : case xs of
             x1 : _ -> go (key x1) xs ky yss
             [] -> yss
         else y : case ys of
            y1 : _ -> go kx xss (key y1) ys
            [] -> xss

       start [] ys = ys
       start xs [] = xs
       start xs@(x : _) ys@(y : _) = go (key x) xs (key y) ys
   in start
