{-|
Module      : Nauty.Graph6.Internal
Description : Internal functions.
Copyright   : (c) Marcelo Garlet Milani, 2026
License     : MIT
Maintainer  : mgmilani@pm.me
Stability   : unstable

This module contains internal functions used by the "Nauty.Graph6" module.
Except for test cases, you should not import this module.
-}

{-# LANGUAGE OverloadedStrings #-}

module Nauty.Graph6.Internal where

import qualified Data.Text.Lazy as T
import qualified Data.Text.Lazy.Encoding as T
import qualified Data.ByteString.Lazy as B
import Control.Monad.Trans.State
import qualified Data.Array.Unboxed as A
import Data.Bits
import Data.Word
import Nauty.Internal.Parsing
import Nauty.Internal.Encoding

-- |A graph represented as an adjacency matrix.
data AdjacencyMatrix = AdjacencyMatrix
  { numberOfVertices :: Word64
  -- |The upper diagonal of the adjacency matrix, represented as a bit string, stored column-by-column.
  , adjacency :: A.UArray Word64 Word8   } deriving (Eq, Show)
  
-- |Whether two vertices are adjacent.
areAdjacent :: AdjacencyMatrix -> Word64 -> Word64 -> Bool
areAdjacent m v u
  | v > u = areAdjacent m u v
  | otherwise = 
    let i = v + (((u) * (u - 1)) `div` 2)
        b = i `div` 8
    in testBit ((adjacency m) A.! b) (7 - ((fromIntegral i) `mod` 8))

-- |Encode a graph into @graph6@ format.
encode :: AdjacencyMatrix -> T.Text
encode m = 
  (encodeNumber $ numberOfVertices m)
  `T.append`
  (encodeMatrix m)

-- |Encode the adjacency matrix.
encodeMatrix :: AdjacencyMatrix -> T.Text
encodeMatrix m = 
  let n = numberOfVertices m
      bits = ((n * (n - 1)) `div` 2)
      lastValidBits = bits `mod` 8
  in
  if n == 1 && [0] == (A.elems (adjacency m)) then
    ""
  else
    T.pack $ map (toEnum . fromIntegral . (+63)) $ encodeVector lastValidBits (A.elems $ adjacency m) 0 6

-- |Create an adjacency matrix from a list of edges.
-- Vertices need to be in the range from @0@ to @n - 1@.
fromEdgeList :: Word64 -- ^ Number of vertices
             -> [(Word64, Word64)] -- ^ List of edges
             -> AdjacencyMatrix
fromEdgeList n es = 
  let m = if n == 1 then 0 else (((n * (n - 1)) `div` 2) - 1) `div` 8
  in
  AdjacencyMatrix
  { numberOfVertices = n
  , adjacency = A.accumArray (.|.) 0 (0, m)
      [ (block, shiftL 1 (7 - (fromIntegral $ bitI `mod` 8)))
      | (v,u) <- es
      , let (v', u') = if v < u then (v,u) else (u,v)
      , let bitI = v' + (((u') * (u' - 1)) `div` 2)
      , let block = bitI `div` 8
      ]
  }

-- |The list of edges of a graph together with the number of vertices.
toEdgeList :: AdjacencyMatrix -> (Word64, [(Word64, Word64)])
toEdgeList m =
  ( numberOfVertices m
  , edges 7 0 1 $ A.elems $ adjacency m)
  where
    edges _ _ _ [] = []
    edges i v u (b:bs)
      | i == -1 = edges 7 v u bs
      | v == u = edges i 0 (u + 1) (b:bs)
      | u == numberOfVertices m = []
      | otherwise = 
        if testBit b i then
          (v,u) : edges (i - 1) (v+1) u (b:bs)
        else
          edges (i - 1) (v+1) u (b:bs)

-- |Parse all graphs in the input text.
-- Graphs are stored one per line.
parse :: T.Text -> [Either T.Text AdjacencyMatrix]
parse t = 
  let t' = header ">>graph6<<" t
  in map graph $ T.lines t'

-- |Parse a single graph in @graph6@ format.
graph :: T.Text -> Either T.Text AdjacencyMatrix
graph t = (flip evalStateT) (T.encodeUtf8 t) $ do
  n <- parseNumber
  parseMatrix n

-- |Parse the adjacency matrix of a graph.
parseMatrix :: Word64 -> StateT B.ByteString (Either T.Text) AdjacencyMatrix
parseMatrix n = do
  v <- parseVector ((n * (n - 1)) `div` 2)
  let m = if n == 1 then 0 else (((n * (n - 1)) `div` 2) - 1) `div` 8
  return $ AdjacencyMatrix
    { numberOfVertices = n
    , adjacency = A.array (0, m )
                          $ zip [0..] $ B.unpack v
    }

