# kb-text-shape

Haskell wrapper for the `kb_text_shape.h` unicode segmentation and shaping library.

[kb\_text\_shape.h]: https://github.com/JimmyLefevre/kb/blob/main/kb_text_shape.h

> ```
>   Your text       A        Text runs with         B       Sequence of glyphs      C
>   (Probably ------------> uniform direction ------------> ready to rasterize ------------> Pixels
>   UTF-8)                    and script
> ```
> We call arrow A text segmentation, arrow B text shaping, and arrow C rasterization.
> This library does A and B.

```haskell
import KB.Text.Shape where qualified as TextShape

main = do
  TextShape.withContext \ctx -> do
    _fallback <- TextShape.pushFontFromFile ctx "test/NotoSans-Regular.ttf" 0
    _main <- TextShape.pushFontFromFile ctx "test/Ubuntu-R.ttf" 0
    results <- TextShape.run ctx do
      TextShape.text_ "A bunch of characters"
      TextShape.char_ '!'
    forM_ results \(run, glyphs) ->
      forM_ glyphs \glyph ->
        print (run.font, glyph.id, glyph.gpos.advanceX)
```

## Font blobs

The library has an internal "blob" format for the pre-processed font data for leaner files and faster loading.

```haskell
import KB.Text.Shape.Font qualified as Font

main = do
  ttfData <- ByteString.readFile "test/Ubuntu-R.ttf"
  putStrLn $ "Distilling " <> show (ByteString.length ttfData `div` 1024) <> "Kb of TTF"
  blobData <- Font.extractBlob ttfData 0
  putStrLn $ "Distilled into " <> show (ByteString.length blobData `div` 1024) <> "Kb blob"
  ByteString.writeFile "test/Ubuntu-R.kbts" blobData
```

The savings are modest, but can be improved even further with compression.

```
Distilling 1047Kb of TTF
Distilled into 618Kb blob
```
